package cn.iinti.majora3.adr.ui.screen

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.NetworkCheck
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.ElevatedCard
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.livedata.observeAsState
import androidx.compose.runtime.mutableStateListOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import cn.iinti.majora3.adr.ConfigItem
import cn.iinti.majora3.adr.R
import cn.iinti.majora3.adr.UILogListener
import cn.iinti.majora3.adr.clearUILogListener
import cn.iinti.majora3.adr.clientGroup
import cn.iinti.majora3.adr.extra
import cn.iinti.majora3.adr.majora3App
import cn.iinti.majora3.adr.registerUILogListener
import cn.iinti.majora3.adr.serverEndpoint
import cn.iinti.majora3.adr.ui.component.TextEdit
import com.ramcosta.composedestinations.annotation.Destination
import com.ramcosta.composedestinations.annotation.RootNavGraph
import com.ramcosta.composedestinations.navigation.DestinationsNavigator

@RootNavGraph(start = true)
@Destination
@Composable
fun HomeScreen(navigator: DestinationsNavigator) {
    Scaffold(topBar = {
        TopBar()
    }) { innerPadding ->
        val networkState by majora3App.hasNetwork.observeAsState(true)
        Column(
            modifier = Modifier
                .padding(innerPadding)
                .padding(horizontal = 8.dp)
                .verticalScroll(rememberScrollState()),
            verticalArrangement = Arrangement.spacedBy(16.dp)
        ) {
            if (networkState) {
                MainPanel()
                LogPanel()
            } else {
                NoNetworkInfoCard()
            }
            Spacer(Modifier)
        }
    }
}


@Composable
private fun NoNetworkInfoCard() {
    ElevatedCard(
        colors = CardDefaults.elevatedCardColors(containerColor = run {
            MaterialTheme.colorScheme.errorContainer
        })
    ) {
        Row(
            modifier = Modifier
                .fillMaxWidth()
                .padding(24.dp),
            verticalAlignment = Alignment.CenterVertically
        ) {
            Icon(
                Icons.Outlined.NetworkCheck, "网络错误"
            )
            Column(Modifier.padding(start = 20.dp)) {
                Text(
                    text = "网络错误",
                    style = MaterialTheme.typography.titleMedium
                )
                Spacer(Modifier.height(4.dp))
                Text(
                    text = "当前手机没有网络",
                    style = MaterialTheme.typography.bodyMedium
                )
            }
        }
    }
}


@OptIn(ExperimentalMaterial3Api::class)
@Composable
private fun TopBar(
) {
    TopAppBar(title = { Text(stringResource(R.string.app_name)) })
}

@Composable
fun LogPanel() {
    val logMessages = remember { mutableStateListOf<String>() }

    // Register global log listener
    val logListener = remember {
        object : UILogListener {
            override fun onLogMessage(message: String) {
                if (logMessages.size >= 20) {
                    logMessages.removeFirst()
                }
                logMessages.add(message)
            }
        }
    }

    // Unregister listener when composable is disposed
    DisposableEffect(Unit) {
        registerUILogListener(logListener)
        onDispose {
            clearUILogListener()
        }
    }

    ElevatedCard(
        modifier = Modifier.fillMaxHeight(),
        colors = CardDefaults.elevatedCardColors(containerColor = run {
            MaterialTheme.colorScheme.primaryContainer
        })
    ) {
        Column(
            modifier = Modifier
                .fillMaxWidth()
                .padding(16.dp)
                .height(300.dp)
                .verticalScroll(rememberScrollState()),
            verticalArrangement = Arrangement.spacedBy(8.dp)
        ) {
            Text(
                text = logMessages.joinToString("\n"),
                style = MaterialTheme.typography.bodyMedium
            )
        }
    }

}


@Composable
private fun MainPanel() {
    ElevatedCard(
        colors = CardDefaults.elevatedCardColors(containerColor = run {
            MaterialTheme.colorScheme.primaryContainer
        })
    ) {
        Column(
            modifier = Modifier
                .padding(16.dp)
        ) {
            @Composable
            fun ConfigCardItem(label: String, configItem: ConfigItem) {
                TextEdit(
                    label = label,
                    text = configItem.configValue,
                ) { value ->
                    configItem.updateConfigValue(value.trim());
                }
            }

            ConfigCardItem("server", serverEndpoint)
            ConfigCardItem("group", clientGroup)
            ConfigCardItem("extra", extra)
        }
    }


}





