package cn.iinti.majora3.sdk.proto;

import io.netty.buffer.ByteBuf;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.Getter;

/**
 * session创建请求，他由服务器发起，客户端负责执行，他包括三种类型
 *
 * <ul>
 *     <li>connect: 通过连接一个远端服务器创建通道</li>
 *     <li>pty: 创建伪终端通道，即发起一个命令行terminal创建请求（仅声明支持pty的客户端可能发起本情况）</li>
 *     <li>proxyS5: 通过socks5代理的方式创建一个网络通道（通过majora实现内网环境下多个网络穿透的场景使用本功能）</li>
 * </ul>
 */
@Data
@AllArgsConstructor
public abstract class SessionInit implements IProto {
    /**
     * 请求流水号，每个流水号代表了一个session，他是服务端生产和管理
     */
    private int sessionId;

    /**
     * 超时时间，超时后将会返回隧道创建失败
     */
    private final int openTimeout;

    /**
     * traceId，此字段主要用于流程调试，当此字段有值时，
     * 代表本次session需要打印全流程的追踪日志，这样服务器和客户端可以对照调试日志。
     * <p>
     * 由于majora是代理IP框架，存在大量网络请求，一般情况下流量转发日志不会全部打印，而是根据服务器的滑动窗口实现采样
     */
    private final String sessionTraceId;

    private SessionInit(ByteBuf in) {
        this.sessionId = IO.readInt(in);
        this.openTimeout = IO.readInt(in);
        this.sessionTraceId = IO.readString(in);
    }

    @Override
    public void writeTo(ByteBuf out) {
        IO.writeInt(out, sessionId);
        IO.writeInt(out, openTimeout);
        IO.writeString(out, sessionTraceId);
    }

    public abstract String desc();

    /**
     * TCP隧道创建请求
     */
    @Getter
    public static class TCP extends SessionInit {
        public static final ByteBufCreator<TCP> CREATOR =
                new ByteBufCreator<TCP>(TCP.class) {
                    @Override
                    public TCP create(ByteBuf in) {
                        return new TCP(in);
                    }
                };
        private final String host;
        private final int port;

        public TCP(int openTimeout, String sessionTraceId,
                   String host, int port) {
            super(-1, openTimeout, sessionTraceId);
            this.host = host;
            this.port = port;
        }

        private TCP(ByteBuf in) {
            super(in);
            this.host = IO.readString(in);
            this.port = in.readInt();
        }

        @Override
        public void writeTo(ByteBuf out) {
            super.writeTo(out);
            IO.writeString(out, host);
            IO.writeInt(out, port);
        }

        @Override
        public String desc() {
            return getSessionId() + " tcp from " + host + ":" + port;
        }
    }

    /**
     * PTY创建请求
     */
    public static class Pty extends SessionInit {
        public static final ByteBufCreator<Pty> CREATOR =
                new ByteBufCreator<Pty>(Pty.class) {

                    @Override
                    public Pty create(ByteBuf in) {
                        return new Pty(in);
                    }
                };

        public Pty(ByteBuf in) {
            super(in);
        }

        public Pty(int openTimeout, String sessionTraceId) {
            super(-1, openTimeout, sessionTraceId);
        }

        @Override
        public String desc() {
            return getSessionId() + " pty";
        }
    }
}
