package cn.iinti.majora3.sdk.proto;

import io.netty.buffer.ByteBuf;
import lombok.AllArgsConstructor;
import lombok.Data;

/**
 * 客户端注册请求
 */
@Data
@AllArgsConstructor
public class ClientRegister implements IProto {

    public static final ByteBufCreator<ClientRegister> CREATOR =
            new ByteBufCreator<ClientRegister>(ClientRegister.class) {
                @Override
                public ClientRegister create(ByteBuf in) {
                    return new ClientRegister(in);
                }
            };

    /**
     * 客户端id，系统看待客户端的唯一ID
     */
    private final String clientId;

    /**
     * 额外属性，客户端配置的一个额外的值，主要用于支持指令IP池。
     * 请注意，extra内容在框架层面没有定义格式，它实际上是用户配置的指令IP池扩展脚本对这个字段进行自定义的解析。
     * <ul>
     *     <li>配置国家地区，服务器提供根据国家/地区进行IP路由的功能</li>
     *     <li>配置isp、终端类型、供应商等信息，服务器根据这些信息进行IP池路由</li>
     * </ul>
     */
    private final String extra;

    /**
     * 加密密钥，此字段用于对流量进行加密,当majora运行在各种防火墙之上时，如果为明文流量，则非常容易被防火墙拦截
     * 我们通过客户端发送随机密码，实现对流量的加密，加密方式为根据给定的密码，进行有符号int的异或。
     * <p>
     * 请注意majora本身不提供"翻墙"能力，如此majora不会将重心放到如何隐蔽流量，以及如何实现隐私保护等功能。
     * <p>
     * 请注意，若此字段为0，则代表不执行加密操作
     */
    private final int encryptXor;

    /**
     * 分组，在客户端安装部署时，运维同学可以提供一个分组，主要方便用户运维管理，此字段可以为空，
     * 后台管理员可以在后台重新填写这个字段，此时group管理将会在后端数据库中维护
     */
    private final String endpointGroup;


    /**
     * 当前客户端是否支持pty，支持pty的终端，将会按照majora要求，提供远程shell的功能。这样运维操作可以直接在web terminal上执行命令行
     */
    private final boolean supportPty;

    /**
     * 当前客户端是否支持重播，支持重播的终端，可以被平台管理发起重播
     * <p>
     * 发起重播指通过网卡重连的方式获得新的出口IP，这样majora就可以获得新的IP资源
     * <ul>
     *     <li>对于vps场景，一版支持通过重播的方式更换IP</li>
     *     <li>对于拥有root权限的手机，支持通过飞行模式切换获得新的出口IP</li>
     *     <li>对于普通客户端集成的情况，为了减少普通用户干扰，一版不建议支持本功能</li>
     * </ul>
     * 请注意：
     * <ul>
     *     <li>在majora3时代，重播调度发起统一发生在服务端的统一控制，不在建议客户端主动管理重播</li>
     *     <li>客户端在重播完成后，需要主动完成网络环境探测，并且恢复到服务器的连接，若长时间连接不成功，客户端可以执行各种策略，包括：再次重播、重启客户端系统等</li>
     *     <li>和majora2不同的时，majoraV3将不再支持android客户端通过shizhuku实现飞行模式重播，而是一律要求获得root权限</li>
     * </ul>
     */
    private final boolean supportRedial;

    /**
     * 客户端所在系统平台：windows、MacOs、Linux、Android
     */
    private final String platform;

    /**
     * 客户端所在系统的版本，如windows7
     */
    private final String osVersion;

    /**
     * 客户端版本，由客户端开发者自行管理
     */
    private final String clientVersion;

    /**
     * 出口IP，对于出入不一致双网卡场景或者二级代理接入场景，出口IP需要端上上报上来
     * 当然，大部分情况下，出口IP有majora服务端自动解析
     */
    private final String publicIp;

    @Override
    public void writeTo(ByteBuf out) {
        IO.writeString(out, clientId);
        IO.writeString(out, extra);
        IO.writeInt(out, encryptXor);
        IO.writeString(out, endpointGroup);
        IO.writeBoolean(out, supportPty);
        IO.writeBoolean(out, supportRedial);
        IO.writeString(out, platform);
        IO.writeString(out, osVersion);
        IO.writeString(out, clientVersion);
        IO.writeString(out, publicIp);
    }

    private ClientRegister(ByteBuf in) {
        clientId = IO.readString(in);
        extra = IO.readString(in);
        encryptXor = IO.readInt(in);
        endpointGroup = IO.readString(in);
        supportPty = IO.readBoolean(in);
        supportRedial = IO.readBoolean(in);
        platform = IO.readString(in);
        osVersion = IO.readString(in);
        clientVersion = IO.readString(in);
        publicIp = IO.readString(in);
    }

}
