package cn.iinti.majora3.sdk.codec;

import cn.iinti.majora3.sdk.MajoraLogger;
import cn.iinti.majora3.sdk.proto.IProto;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.ByteToMessageCodec;

import java.util.List;

import static io.netty.buffer.ByteBufUtil.appendPrettyHexDump;
import static io.netty.util.internal.StringUtil.NEWLINE;

public class MajoraCodeC extends ByteToMessageCodec<IProto> {
    private final MajoraLogger codeCLogger;
    private final int encryptXor;

    public MajoraCodeC(MajoraLogger codeCLogger, int encryptXor) {
        this.codeCLogger = codeCLogger;
        this.encryptXor = encryptXor;
    }

    /**
     * 打印ByteBuf的好工具
     */
    public static String formatByteBuf(ChannelHandlerContext ctx, String eventName, ByteBuf msg) {
        String chStr = ctx == null ? "debug" : ctx.channel().toString();
        if (msg == null) {
            msg = Unpooled.EMPTY_BUFFER;
        }
        int length = msg.readableBytes();
        if (length == 0) {
            return chStr + ' ' + eventName + ": 0B";
        } else {
            int outputLength = chStr.length() + 1 + eventName.length() + 2 + 10 + 1;

            int rows = length / 16 + (length % 15 == 0 ? 0 : 1) + 4;
            int hexDumpLength = 2 + rows * 80;
            outputLength += hexDumpLength;

            StringBuilder buf = new StringBuilder(outputLength);
            buf.append(chStr).append(' ').append(eventName).append(": ").append(length).append('B');

            buf.append(NEWLINE);
            appendPrettyHexDump(buf, msg);
            return buf.toString();
        }
    }

    private static void xor(ByteBuf payload, int len, int xor) {
        int intIndex = len & 0xFFFFFFFC;
        int i = 0;
        for (; i < intIndex; i += 4) {
            int value = payload.getInt(i);
            payload.setInt(i, value ^ xor);
        }
        for (; i < len; i++) {
            byte value = payload.getByte(i);
            payload.setByte(i, value ^ xor);
        }
    }

    public static class CodeCError extends Exception {
        public CodeCError(String message) {
            super(message);
        }
    }


    @Override
    protected void encode(ChannelHandlerContext ctx, IProto msg, ByteBuf out) throws CodeCError {
        PackageType packageType = PackageType.of(msg.getClass());
        if (packageType == null) {
            throw new CodeCError("can not resolve PackageType of " + msg.getClass());
        }
        byte typeCode = packageType.getCode();
        out.writeByte(typeCode);

        ByteBuf payloadContent = ctx.alloc().buffer();
        try {
            // decode into temp buffer
            msg.writeTo(payloadContent);

            // write length
            out.writeInt(payloadContent.readableBytes());

            // encrypt
            // 这里会导致所有数据都会经过CPU走一遍，在极限情况下，可能导致cpu资源消耗，
            // 不过分析了一下netty的代码 ，在netty内部的编解码过程对数据全量遍历的场景还是挺多的，
            // DeepSeek也说一般情况下不用考虑这里的性能问题
            // 所以这里我们期待java的JIT和多核服务器的情况下应该不会有性能瓶颈了
            xor(payloadContent, payloadContent.readableBytes(), encryptXor);

            // flush to buffer
            out.writeBytes(payloadContent);
        } finally {
            payloadContent.release();
        }

        if (codeCLogger.enable()) {
            String logMsg = formatByteBuf(ctx, "majora encode data", out);
            codeCLogger.log(() -> logMsg);
        }
    }


    private boolean decodeOnce(ChannelHandlerContext ctx, ByteBuf in, List<Object> out) throws Exception {
        byte typeCode = in.readByte();
        PackageType packageType = PackageType.of(typeCode);
        if (packageType == null) {
            throw new CodeCError("unknown package type: " + typeCode);
        }

        int payloadLength = in.readInt();
        if (in.readableBytes() < payloadLength) {
            return false;
        }

        ByteBuf content = ctx.alloc().buffer(payloadLength);
        // todo 这里其实进行了两次数据遍历，虽然第一次可能被操作系统优化了。但是理论上可以优化为只遍历一次
        in.readBytes(content, payloadLength);

        // decrypt
        xor(content, payloadLength, encryptXor);

        codeCLogger.log(() -> "decoder Got: " + packageType);
        out.add(packageType.getByteBufCreator().create(content));
        // content不会传递到下一个pipeline，所以需要释放，否则会有内存泄漏
        content.release();
        return true;
    }

    @Override
    protected void decode(ChannelHandlerContext ctx, ByteBuf in, List<Object> out) throws Exception {
        if (codeCLogger.enable()) {
            String logMsg = formatByteBuf(ctx, "majora decode data", in);
            codeCLogger.log(() -> logMsg);
        }

        while (in.isReadable()) {
            if (in.readableBytes() < 5) {
                // type + frameLength = 1 + 4
                break;
            }
            in.markReaderIndex();
            if (!decodeOnce(ctx, in, out)) {
                if (codeCLogger.enable()) {
                    String logMsg = "not enough bytes to read: " + in;
                    codeCLogger.log(() -> logMsg);
                }
                in.resetReaderIndex();
                break;
            }
        }
    }
}
