package cn.iinti.majora3.sdk.client;

import cn.iinti.majora3.sdk.MajoraLogger;
import cn.iinti.majora3.sdk.codec.MagicHandSharker;
import cn.iinti.majora3.sdk.codec.MajoraCodeC;
import cn.iinti.majora3.sdk.codec.PackageType;
import cn.iinti.majora3.sdk.proto.*;
import io.netty.bootstrap.Bootstrap;
import io.netty.buffer.ByteBuf;
import io.netty.channel.*;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.SocketChannel;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.handler.timeout.IdleStateEvent;
import io.netty.handler.timeout.IdleStateHandler;

import java.util.Random;

public class MajoraConnection extends SimpleChannelInboundHandler<IProto> {
    private final String host;
    private final int port;
    private final int connectionTimeout;
    private final int encryptXor = new Random().nextInt();
    private final MajoraClient majoraClient;
    private final MajoraLogger logger;
    private final boolean logCodeC;


    private final NioEventLoopGroup workerGroup;
    private final ConnectionListener connectionListener;

    private final Channel majoraServerChannel;

    private final SessionManager sessionManager;
    private final CtrManager ctrManager;

    public MajoraConnection(String host, int port, int connectionTimeout,
                            MajoraClient majoraClient,
                            NioEventLoopGroup clientLoopGroup,
                            ConnectionListener listener,
                            boolean logCodeC
    ) {
        this.host = host;
        this.port = port;
        this.logCodeC = logCodeC;
        this.connectionTimeout = connectionTimeout;
        this.majoraClient = majoraClient;
        this.logger = majoraClient.getLogger();
        this.workerGroup = clientLoopGroup;
        this.connectionListener = listener;
        this.sessionManager = new SessionManager(majoraClient, this);
        this.ctrManager = new CtrManager(majoraClient, this);
        this.majoraServerChannel = connect();
    }

    private Channel connect() {
        Bootstrap bootstrap = new Bootstrap().group(workerGroup)
                .channel(NioSocketChannel.class)
                .option(ChannelOption.CONNECT_TIMEOUT_MILLIS, connectionTimeout)
                .handler(new ChannelInitializer<SocketChannel>() {

                    @Override
                    public void initChannel(SocketChannel ch) {
                        MajoraLogger majoraLogger = logCodeC ? logger : MajoraLogger.nop;
                        IdleStateHandler idleStateHandler = new IdleStateHandler(
                                MajoraClient.HEARTBEAT_INTERVAL + 12, 0, 0
                        );
                        ch.pipeline().addLast(
                                idleStateHandler, // 超时控制
                                new MagicHandSharker(ch, majoraLogger), // 处理第一个magic
                                new MajoraCodeC(majoraLogger, encryptXor),  // 编解码
                                MajoraConnection.this // 业务
                        );

                        ch.closeFuture().addListener((ChannelFutureListener) future -> {
                                    logger.log(() -> "server connection closed");
                                    invokeOnDisconnected();
                                    sessionManager.destroy();
                                }
                        );
                    }
                });
        logger.log(() -> "begin to connect to " + host + ":" + port);
        ChannelFuture channelFuture = bootstrap.connect(host, port);
        channelFuture.addListener((ChannelFutureListener) future -> {
            if (!future.isSuccess()) {
                logger.log(() -> "connect to " + host + ":" + port + " failed ", future.cause());
                invokeOnDisconnected();
                return;
            }
            logger.log(() -> "connect to " + host + ":" + port + " success");
            registerClient(channelFuture.channel());
        });
        return channelFuture.channel();
    }

    private void registerClient(Channel channel) {
        channel.write(Env.magicBuf());

        ByteBuf xorSeedBuf = channel.alloc().buffer(4);
        xorSeedBuf.writeInt(encryptXor);

        channel.write(xorSeedBuf).addListener((ChannelFutureListener) future -> {
            logger.log(() -> "magic send status:" + future.isSuccess());
        });

        boolean supportPty = SessionManager.supportSession(PackageType.SESSION_INIT_PTY);
        boolean supportRedial = CtrManager.supportCtrAction(CtrManager.DEFAULT_ACTION_REDIAL);
        ClientRegister clientRegister = new ClientRegister(majoraClient.getClientId(),
                majoraClient.getExtra(), encryptXor, majoraClient.getEndpointGroup(),
                supportPty, supportRedial,
                Env.platform, Env.osVersion, MajoraClient.CLIENT_VERSION, "");
        logger.log(() -> "send register pkg:" + clientRegister);

        // write register packet
        channel.writeAndFlush(clientRegister).addListener((ChannelFutureListener) future -> {
                    if (!future.isSuccess()) {
                        logger.log(() -> "send register pkg failed ", future.cause());
                        return;
                    }
                    logger.log(() -> "send register pkg success");
                    invokeOnConnected();
                }
        );
    }


    @Override
    protected void channelRead0(ChannelHandlerContext ctx, IProto iProto) throws Exception {
        if (iProto instanceof HeartBeat) {
            logger.log(() -> "heartbeat from server");
            HeartBeat heartBeat = (HeartBeat) iProto;
            ctx.writeAndFlush(heartBeat);
        } else if (iProto instanceof SessionInit) {
            SessionInit sessionInit = (SessionInit) iProto;
            sessionManager.createSession(majoraClient, sessionInit);
        } else if (iProto instanceof SessionSignal.SessionClose) {
            SessionSignal.SessionClose sessionClose = (SessionSignal.SessionClose) iProto;
            sessionManager.onInboundStreamClose(sessionClose.getSessionId());
        } else if (iProto instanceof SessionPayload) {
            SessionPayload sessionPayload = (SessionPayload) iProto;
            sessionManager.onInboundStreamData(sessionPayload);
        } else if (iProto instanceof CtrReq) {
            CtrReq ctrReq = (CtrReq) iProto;
            ctrManager.handleCtrReq(ctrReq);
        } else {
            logger.log(() -> "unknown proto: " + iProto.getClass());
        }
    }

    @Override
    public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
        if (evt instanceof IdleStateEvent) {
            logger.log(() -> "channel idle, close to restart");
            ctx.close();
            return;
        }
        super.userEventTriggered(ctx, evt);
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
        logger.log(() -> "exception caught: ", cause);
        ctx.close();
    }

    private boolean connectedInvoked = false;

    private void invokeOnConnected() {
        if (connectedInvoked) {
            return;
        }
        majoraClient.doOnMainThead(() -> {
            connectedInvoked = true;
            connectionListener.onConnected(MajoraConnection.this);
        });
    }

    private boolean disconnectInvoked = false;

    private void invokeOnDisconnected() {
        if (disconnectInvoked) {
            return;
        }
        majoraClient.doOnMainThead(() -> {
            disconnectInvoked = true;
            connectionListener.onDisconnected(MajoraConnection.this);
        });
    }

    public interface ConnectionListener {
        void onConnected(MajoraConnection majoraConnection);

        void onDisconnected(MajoraConnection majoraConnection);
    }

    public boolean isActive() {
        return majoraServerChannel.isActive();
    }

    public void close() {
        majoraServerChannel.close();
    }

    void writeToMajora(IProto iProto) {
        if (majoraServerChannel.isActive()) {
            majoraServerChannel.writeAndFlush(iProto);
        }
    }
}
