package cn.iinti.majora3.sdk.client;

import org.apache.commons.lang3.StringUtils;

import cn.iinti.majora3.sdk.MajoraLogger;
import io.netty.channel.EventLoop;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.util.concurrent.DefaultThreadFactory;
import lombok.Getter;
import lombok.Setter;

import java.util.concurrent.TimeUnit;

public class MajoraClient implements MajoraConnection.ConnectionListener {
    public static final String CLIENT_VERSION = "1.0.0";
    public static final int HEARTBEAT_INTERVAL = 30;
    @Getter
    private final String clientId;
    private final String serverHost;
    private final int serverPort;
    @Getter
    @Setter
    private String endpointGroup;

    @Getter
    @Setter
    private String extra;

    @Setter
    @Getter
    // default is nop
    private MajoraLogger logger = MajoraLogger.nop;

    /**
     * 默认30s时间到服务器的连接超时
     */
    @Setter
    private int connectionTimeout = 30_000;

    @Setter
    private boolean logCodec = false;

    private final int[] reconnectWaitSlot = new int[]{
            10, 10, 15, 15, 15, 20, 30, 30, 45
    };
    private int failedCount;

    private MajoraConnection currentConnection;

    /**
     * 线程组，这是重型对象，需要放到全局共享
     */
    private static final NioEventLoopGroup workerGroup = new NioEventLoopGroup(
            0, new DefaultThreadFactory("majora-client")
    );

    /**
     * 设定线程组第一个线程为主线程，主线程的作用是将一些可能存在并发竞争的操作规约在相同线程下，
     * 这样可以避免存在操作紊乱
     */
    private static final EventLoop mainLoop = workerGroup.next();

    private boolean isConnecting = false;

    private boolean destroyed = false;


    public MajoraClient(String serverHost, int serverPort, String clientId) {
        this.clientId = clientId;
        this.serverHost = serverHost;
        this.serverPort = serverPort;
    }

    public boolean matchConfig(String serverHost, int serverPort, String endpointGroup
            , String extra) {
        return serverHost.equals(this.serverHost) && serverPort == this.serverPort
                && StringUtils.equals(this.endpointGroup, endpointGroup)
                && StringUtils.equals(this.extra, extra);
    }

    public void start() {
        if (destroyed) {
            throw new IllegalStateException("the MajoraClient is destroied");
        }
        doOnMainThead(() -> {
            if (destroyed) {
                return;
            }
            logger.log(() -> "begin start connecting");
            if (currentConnection != null && currentConnection.isActive()) {
                logger.log(() -> "current connection always active,skip connect");
                return;
            }
            if (isConnecting) {
                logger.log(() -> "there is already connecting");
                return;
            }
            isConnecting = true;
            new MajoraConnection(serverHost, serverPort,
                    connectionTimeout, this, workerGroup,
                    this, logCodec
            );
        });
    }

    public void reConnect() {
        doOnMainThead(() -> {
            if (destroyed) {
                return;
            }
            if (currentConnection == null || !currentConnection.isActive()) {
                return;
            }
            currentConnection.close();
        });
    }

    /**
     * 当需要关闭连接的时候，调用此方法,需要注意的是，此方法调用后，MajoraClient实例就销毁了，切不在可以使用
     */
    public void destroy() {
        doOnMainThead(() -> {
            destroyed = true;
            if (currentConnection != null) {
                currentConnection.close();
            }
        });
    }

    public void doOnMainThead(Runnable runnable) {
        doOnMainThead0(() -> {
            try {
                runnable.run();
            } catch (Exception e) {
                logger.log(() -> "execute main loop error", e);
            }
        });
    }

    public static void doOnMainThead0(Runnable runnable) {
        if (mainLoop.inEventLoop()) {
            runnable.run();
        } else {
            mainLoop.execute(runnable);
        }
    }

    @Override
    public void onConnected(MajoraConnection majoraConnection) {
        logger.log(() -> "connection established");
        isConnecting = false;
        failedCount = 0;
        currentConnection = majoraConnection;
    }

    @Override
    public void onDisconnected(MajoraConnection majoraConnection) {
        isConnecting = false;
        int waitTime = failedCount < reconnectWaitSlot.length ? reconnectWaitSlot[failedCount] : 60;
        logger.log(() -> "connection disconnected,will wait for " + waitTime + " seconds to reconnect");
        if (!destroyed) {
            mainLoop.schedule(this::start, waitTime, TimeUnit.SECONDS);
        }
    }
}
