package cn.iinti.majora3.sdk.client;

import cn.iinti.majora3.sdk.proto.CtrReq;
import cn.iinti.majora3.sdk.proto.CtrResp;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;

import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

public class CtrManager {
    /**
     * 内置指令，执行一个命令行，所有终端都应该支持
     */
    public static final String DEFAULT_ACTION_EXECUTE_CMD = "executeCMD";
    /**
     * 内置指令，执行重播操作，客户端根据自己的情况实现，
     * 同时由于他是服务器和终端的默认约定，所以重播的action名字需要固定为确定字符串
     */
    public static final String DEFAULT_ACTION_REDIAL = "redial";

    private static final Executor sharedThread = Executors.newSingleThreadExecutor();
    private static final Map<String, CtrHandler> handlers = new HashMap<>();

    public static void execute(Runnable runnable) {
        sharedThread.execute(runnable);
    }

    public static void registerCtrHandler(String action, CtrHandler handler) {
        handlers.put(action, handler);
    }

    public static boolean supportCtrAction(String action) {
        return handlers.containsKey(action);
    }

    public void handleCtrReq(CtrReq ctrReq) {
        Response response = new Response(ctrReq);
        String ctr = ctrReq.getCtr();
        CtrHandler ctrHandler = handlers.get(ctr);
        if (ctrHandler == null) {
            majoraClient.getLogger().log(() -> "no ctr handler for " + ctrReq.getCtr());
            response.failed("no ctr handler for " + ctrReq.getCtr());
            return;
        }
        try {
            majoraClient.getLogger().log(() -> "receive cmd:" + ctrReq.getCtr());
            ctrHandler.handle(ctrReq.getRequestMap(), response);
        } catch (Throwable t) {
            majoraClient.getLogger().log(() -> "ctr handle error", t);
            response.failed(t.getMessage());
        }
    }


    static {
        registerCtrHandler(DEFAULT_ACTION_EXECUTE_CMD, new ExecuteCmdHandler());
    }

    public CtrManager(MajoraClient majoraClient, MajoraConnection mConnection) {
        this.majoraClient = majoraClient;
        this.mConnection = mConnection;
    }

    private final MajoraClient majoraClient;
    private final MajoraConnection mConnection;

    public interface CtrHandler {
        void handle(Map<String, String> request, Response response);
    }


    public class Response {
        private final AtomicBoolean called = new AtomicBoolean(false);

        private final CtrReq req;

        public Response(CtrReq req) {
            this.req = req;
        }

        public void success(Map<String, String> resp) {
            if (!called.compareAndSet(false, true)) {
                return;
            }
            CtrResp ctrResp = new CtrResp(req.getRequestSessionId(), "", resp);
            mConnection.writeToMajora(ctrResp);
        }

        public void failed(String failedMessage) {
            if (!called.compareAndSet(false, true)) {
                return;
            }
            CtrResp ctrResp = new CtrResp(req.getRequestSessionId(), failedMessage, new HashMap<>());
            mConnection.writeToMajora(ctrResp);
        }
    }

    private static class ExecuteCmdHandler implements CtrHandler {

        @Override
        public void handle(Map<String, String> request, Response response) {
            String cmd = request.get("cmd");
            if (StringUtils.isBlank(cmd)) {
                response.failed("no param:{cmd} presented");
                return;
            }

            execute(() -> {
                try {
                    doExecuteCmd(cmd, response);
                } catch (Exception e) {
                    response.failed(e.getMessage());
                }
            });
        }

        private static void doExecuteCmd(String cmd, Response response) throws Exception {
            Process process = Runtime.getRuntime().exec(cmd);
            boolean finished = process.waitFor(2, TimeUnit.MINUTES);
            if (!finished) {
                response.failed("command execution timeout");
                return;
            }
            HashMap<String, String> result = new HashMap<>();
            result.put("error", IOUtils.toString(process.getErrorStream(), StandardCharsets.UTF_8));
            result.put("output", IOUtils.toString(process.getInputStream(), StandardCharsets.UTF_8));
            response.success(result);
        }
    }
}
